/***************************************************************
 * Name:      complex_setMain.cpp
 * Purpose:   Code for Application Frame
 * Author:    Petar ()
 * Created:   2025-01-28
 * Copyright: Petar ()
 * License:
 **************************************************************/

#include "complex_setMain.h"
#include <wx/msgdlg.h>
#include <wx/filedlg.h>
#include <wx/file.h>
#include <wx/dcclient.h>
#include <wx/dcbuffer.h>
#include <algorithm>
#include <cmath>

//(*InternalHeaders(complex_setFrame)
#include <wx/intl.h>
#include <wx/string.h>
//*)

//helper functions
enum wxbuildinfoformat {
    short_f, long_f };

wxString wxbuildinfo(wxbuildinfoformat format)
{
    wxString wxbuild(wxVERSION_STRING);

    if (format == long_f )
    {
#if defined(__WXMSW__)
        wxbuild << _T("-Windows");
#elif defined(__UNIX__)
        wxbuild << _T("-Linux");
#endif

#if wxUSE_UNICODE
        wxbuild << _T("-Unicode build");
#else
        wxbuild << _T("-ANSI build");
#endif // wxUSE_UNICODE
    }

    return wxbuild;
}

//(*IdInit(complex_setFrame)
const long complex_setFrame::ID_STATICTEXT1 = wxNewId();
const long complex_setFrame::ID_RETEXT1 = wxNewId();
const long complex_setFrame::ID_STATICTEXT2 = wxNewId();
const long complex_setFrame::ID_IMTEXT1 = wxNewId();
const long complex_setFrame::ID_NEW1 = wxNewId();
const long complex_setFrame::ID_SAVE1 = wxNewId();
const long complex_setFrame::ID_DELETE1 = wxNewId();
const long complex_setFrame::ID_STATICTEXT3 = wxNewId();
const long complex_setFrame::ID_RETEXT2 = wxNewId();
const long complex_setFrame::ID_STATICTEXT4 = wxNewId();
const long complex_setFrame::ID_IMTEXT2 = wxNewId();
const long complex_setFrame::ID_NEW2 = wxNewId();
const long complex_setFrame::ID_SAVE2 = wxNewId();
const long complex_setFrame::ID_DELETE2 = wxNewId();
const long complex_setFrame::ID_LISTBOX1 = wxNewId();
const long complex_setFrame::ID_LISTBOX2 = wxNewId();
const long complex_setFrame::ID_DRAWPANEL1 = wxNewId();
const long complex_setFrame::ID_MENUITEM1 = wxNewId();
const long complex_setFrame::idMenuAbout = wxNewId();
const long complex_setFrame::ID_STATUSBAR1 = wxNewId();
//*)

BEGIN_EVENT_TABLE(complex_setFrame,wxFrame)
    //(*EventTable(complex_setFrame)
    EVT_CLOSE(complex_setFrame::OnClose)
    EVT_LISTBOX(wxID_ANY, complex_setFrame::OnSet1Select)
    EVT_LISTBOX(wxID_ANY, complex_setFrame::OnSet2Select)
    //EVT_PAINT(complex_setFrame::OnPanelPaint)
    //*)
END_EVENT_TABLE()

bool ComplexSet::Add(const std::complex<double>& elem) {
    return elements.insert(elem).second;
}

bool ComplexSet::Remove(const std::complex<double>& elem) {
    return elements.erase(elem) > 0;
}

bool ComplexSet::Contains(const std::complex<double>& elem) const {
    return elements.find(elem) != elements.end();
}

void ComplexSet::LoadFromFile(const wxString& filename) {
    elements.clear();
    wxFile file;
    if (file.Open(filename, wxFile::read)) {
        wxString content;
        file.ReadAll(&content);
        wxArrayString lines = wxSplit(content, '\n');
        for (const wxString& line : lines) {
            wxArrayString parts = wxSplit(line, ' ');
            if (parts.size() >= 2) {
                double re, im;
                if (parts[0].ToDouble(&re) && parts[1].ToDouble(&im)) {
                    elements.insert(std::complex<double>(re, im));
                }
            }
        }
        file.Close();
    }
}

void ComplexSet::SaveToFile(const wxString& filename) const {
    wxFile file;
    if (file.Open(filename, wxFile::write)) {
        for (const auto& elem : elements) {
            file.Write(wxString::Format("%g %g\n", elem.real(), elem.imag()));
        }
        file.Close();
    }
}

std::vector<std::complex<double>> ComplexSet::GetElements() const {
    return std::vector<std::complex<double>>(elements.begin(), elements.end());
}

void ComplexSet::GetBounds(double& reMin, double& reMax, double& imMin, double& imMax) const {
    reMin = reMax = imMin = imMax = 0.0;
    if (elements.empty()) return;

    auto it = elements.begin();
    reMin = reMax = it->real();
    imMin = imMax = it->imag();

    for (const auto& elem : elements) {
        reMin = std::min(reMin, elem.real());
        reMax = std::max(reMax, elem.real());
        imMin = std::min(imMin, elem.imag());
        imMax = std::max(imMax, elem.imag());
    }
}

bool ComplexSet::IsEmpty() const {
    return elements.empty();
}

complex_setFrame::complex_setFrame(wxWindow* parent,wxWindowID id)
{
    //(*Initialize(complex_setFrame)
    wxBoxSizer* BoxSizer1;
    wxBoxSizer* BoxSizer2;
    wxBoxSizer* BoxSizer3;
    wxBoxSizer* BoxSizer6;
    wxBoxSizer* BoxSizer7;
    wxFlexGridSizer* FlexGridSizer1;
    wxFlexGridSizer* FlexGridSizer2;
    wxFlexGridSizer* FlexGridSizer3;
    wxFlexGridSizer* FlexGridSizer4;
    wxGridBagSizer* GridBagSizer1;
    wxGridBagSizer* GridBagSizer2;
    wxGridBagSizer* GridBagSizer3;
    wxMenu* Menu1;
    wxMenu* Menu2;
    wxMenuBar* MenuBar1;
    wxMenuItem* MenuItem1;
    wxMenuItem* MenuItem2;
    wxStaticBoxSizer* StaticBoxSizer1;
    wxStaticBoxSizer* StaticBoxSizer2;
    wxStaticBoxSizer* StaticBoxSizer3;
    wxStaticBoxSizer* StaticBoxSizer4;

    Create(parent, wxID_ANY, _("Two Complex Sets"), wxDefaultPosition, wxDefaultSize, wxDEFAULT_FRAME_STYLE, _T("wxID_ANY"));
    GridBagSizer1 = new wxGridBagSizer(0, 0);
    BoxSizer3 = new wxBoxSizer(wxHORIZONTAL);
    StaticBoxSizer1 = new wxStaticBoxSizer(wxHORIZONTAL, this, _("Set1_Elem2"));
    GridBagSizer2 = new wxGridBagSizer(0, 0);
    FlexGridSizer1 = new wxFlexGridSizer(0, 3, 0, 0);
    StaticText1 = new wxStaticText(this, ID_STATICTEXT1, _("Re:"), wxDefaultPosition, wxDefaultSize, 0, _T("ID_STATICTEXT1"));
    FlexGridSizer1->Add(StaticText1, 1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    ReText1 = new wxTextCtrl(this, ID_RETEXT1, _("0"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_RETEXT1"));
    ReText1->SetMinSize(wxDLG_UNIT(this,wxSize(100,15)));
    FlexGridSizer1->Add(ReText1, 1, wxALL|wxEXPAND, 5);
    GridBagSizer2->Add(FlexGridSizer1, wxGBPosition(0, 1), wxDefaultSpan, wxALL|wxALIGN_RIGHT|wxALIGN_TOP, 5);
    FlexGridSizer2 = new wxFlexGridSizer(0, 3, 0, 0);
    StaticText2 = new wxStaticText(this, ID_STATICTEXT2, _("Im:"), wxDefaultPosition, wxDefaultSize, 0, _T("ID_STATICTEXT2"));
    FlexGridSizer2->Add(StaticText2, 1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    ImText1 = new wxTextCtrl(this, ID_IMTEXT1, _("0"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_IMTEXT1"));
    ImText1->SetMinSize(wxDLG_UNIT(this,wxSize(100,15)));
    FlexGridSizer2->Add(ImText1, 1, wxALL|wxEXPAND, 5);
    GridBagSizer2->Add(FlexGridSizer2, wxGBPosition(0, 0), wxDefaultSpan, wxALL|wxALIGN_LEFT|wxALIGN_TOP, 5);
    BoxSizer6 = new wxBoxSizer(wxHORIZONTAL);
    New1 = new wxButton(this, ID_NEW1, _("New"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_NEW1"));
    BoxSizer6->Add(New1, 1, wxALL|wxEXPAND, 5);
    Save1 = new wxButton(this, ID_SAVE1, _("Save"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_SAVE1"));
    BoxSizer6->Add(Save1, 1, wxALL|wxEXPAND, 5);
    GridBagSizer2->Add(BoxSizer6, wxGBPosition(1, 0), wxDefaultSpan, wxALL|wxALIGN_LEFT|wxALIGN_BOTTOM, 5);
    BoxSizer7 = new wxBoxSizer(wxHORIZONTAL);
    Delete1 = new wxButton(this, ID_DELETE1, _("Delete"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_DELETE1"));
    BoxSizer7->Add(Delete1, 1, wxALL|wxEXPAND, 5);
    GridBagSizer2->Add(BoxSizer7, wxGBPosition(1, 1), wxDefaultSpan, wxALL|wxALIGN_RIGHT|wxALIGN_BOTTOM, 5);
    GridBagSizer2->AddGrowableCol(0);
    GridBagSizer2->AddGrowableCol(1);
    GridBagSizer2->AddGrowableRow(0);
    GridBagSizer2->AddGrowableRow(1);
    StaticBoxSizer1->Add(GridBagSizer2, 1, wxALL|wxALIGN_TOP, 5);
    BoxSizer3->Add(StaticBoxSizer1, 1, wxALL|wxEXPAND, 5);
    StaticBoxSizer2 = new wxStaticBoxSizer(wxHORIZONTAL, this, _("Set2_Elem4"));
    GridBagSizer3 = new wxGridBagSizer(0, 0);
    FlexGridSizer3 = new wxFlexGridSizer(0, 3, 0, 0);
    StaticText3 = new wxStaticText(this, ID_STATICTEXT3, _("Re:"), wxDefaultPosition, wxDefaultSize, 0, _T("ID_STATICTEXT3"));
    FlexGridSizer3->Add(StaticText3, 1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    ReText2 = new wxTextCtrl(this, ID_RETEXT2, _("0"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_RETEXT2"));
    ReText2->SetMinSize(wxDLG_UNIT(this,wxSize(100,15)));
    FlexGridSizer3->Add(ReText2, 1, wxALL|wxEXPAND, 5);
    GridBagSizer3->Add(FlexGridSizer3, wxGBPosition(0, 1), wxDefaultSpan, wxALL|wxALIGN_RIGHT|wxALIGN_CENTER_VERTICAL, 5);
    FlexGridSizer4 = new wxFlexGridSizer(0, 3, 0, 0);
    StaticText4 = new wxStaticText(this, ID_STATICTEXT4, _("Im:"), wxDefaultPosition, wxDefaultSize, 0, _T("ID_STATICTEXT4"));
    FlexGridSizer4->Add(StaticText4, 1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
    ImText2 = new wxTextCtrl(this, ID_IMTEXT2, _("0"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_IMTEXT2"));
    ImText2->SetMinSize(wxDLG_UNIT(this,wxSize(100,15)));
    FlexGridSizer4->Add(ImText2, 1, wxALL|wxEXPAND, 5);
    GridBagSizer3->Add(FlexGridSizer4, wxGBPosition(0, 0), wxDefaultSpan, wxALL|wxALIGN_LEFT|wxALIGN_CENTER_VERTICAL, 5);
    BoxSizer1 = new wxBoxSizer(wxHORIZONTAL);
    New2 = new wxButton(this, ID_NEW2, _("New"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_NEW2"));
    BoxSizer1->Add(New2, 1, wxALL|wxEXPAND, 5);
    Save2 = new wxButton(this, ID_SAVE2, _("Save"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_SAVE2"));
    BoxSizer1->Add(Save2, 1, wxALL|wxEXPAND, 5);
    GridBagSizer3->Add(BoxSizer1, wxGBPosition(1, 0), wxDefaultSpan, wxALL|wxALIGN_LEFT|wxALIGN_CENTER_VERTICAL, 5);
    BoxSizer2 = new wxBoxSizer(wxHORIZONTAL);
    Delete2 = new wxButton(this, ID_DELETE2, _("Delete"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_DELETE2"));
    BoxSizer2->Add(Delete2, 1, wxALL|wxEXPAND, 5);
    GridBagSizer3->Add(BoxSizer2, wxGBPosition(1, 1), wxDefaultSpan, wxALL|wxALIGN_RIGHT|wxALIGN_CENTER_VERTICAL, 5);
    GridBagSizer3->AddGrowableCol(0);
    GridBagSizer3->AddGrowableCol(1);
    GridBagSizer3->AddGrowableRow(0);
    GridBagSizer3->AddGrowableRow(1);
    StaticBoxSizer2->Add(GridBagSizer3, 1, wxALL|wxALIGN_TOP, 5);
    BoxSizer3->Add(StaticBoxSizer2, 1, wxALL|wxEXPAND, 5);
    GridBagSizer1->Add(BoxSizer3, wxGBPosition(0, 1), wxDefaultSpan, wxALL|wxEXPAND, 5);
    StaticBoxSizer3 = new wxStaticBoxSizer(wxHORIZONTAL, this, _("Set 1"));
    Set1 = new wxListBox(this, ID_LISTBOX1, wxDefaultPosition, wxDefaultSize, 0, 0, 0, wxDefaultValidator, _T("ID_LISTBOX1"));
    Set1->SetMinSize(wxDLG_UNIT(this,wxSize(50,200)));
    StaticBoxSizer3->Add(Set1, 1, wxALL|wxEXPAND, 5);
    GridBagSizer1->Add(StaticBoxSizer3, wxGBPosition(0, 0), wxGBSpan(3, 1), wxALL|wxEXPAND, 5);
    StaticBoxSizer4 = new wxStaticBoxSizer(wxHORIZONTAL, this, _("Set 2"));
    Set2 = new wxListBox(this, ID_LISTBOX2, wxDefaultPosition, wxDefaultSize, 0, 0, 0, wxDefaultValidator, _T("ID_LISTBOX2"));
    Set2->SetMinSize(wxDLG_UNIT(this,wxSize(50,200)));
    StaticBoxSizer4->Add(Set2, 1, wxALL|wxEXPAND, 5);
    GridBagSizer1->Add(StaticBoxSizer4, wxGBPosition(0, 2), wxGBSpan(3, 1), wxALL|wxEXPAND, 5);
    DrawPanel = new wxPanel(this, ID_DRAWPANEL1, wxDefaultPosition, wxDefaultSize, wxTAB_TRAVERSAL, _T("ID_DRAWPANEL1"));
    GridBagSizer1->Add(DrawPanel, wxGBPosition(1, 1), wxGBSpan(2, 1), wxALL|wxEXPAND, 5);
    GridBagSizer1->AddGrowableCol(0);
    GridBagSizer1->AddGrowableCol(1);
    GridBagSizer1->AddGrowableCol(2);
    GridBagSizer1->AddGrowableRow(0);
    GridBagSizer1->AddGrowableRow(1);
    GridBagSizer1->AddGrowableRow(2);
    SetSizer(GridBagSizer1);
    MenuBar1 = new wxMenuBar();
    Menu1 = new wxMenu();
    MenuItem1 = new wxMenuItem(Menu1, ID_MENUITEM1, _("Quit\tAlt-F4"), _("Quit the application"), wxITEM_NORMAL);
    Menu1->Append(MenuItem1);
    MenuBar1->Append(Menu1, _("&File"));
    Menu2 = new wxMenu();
    MenuItem2 = new wxMenuItem(Menu2, idMenuAbout, _("About\tF1"), _("Show info about this application"), wxITEM_NORMAL);
    Menu2->Append(MenuItem2);
    MenuBar1->Append(Menu2, _("Help"));
    SetMenuBar(MenuBar1);
    StatusBar1 = new wxStatusBar(this, ID_STATUSBAR1, 0, _T("ID_STATUSBAR1"));
    int __wxStatusBarWidths_1[1] = { -1 };
    int __wxStatusBarStyles_1[1] = { wxSB_NORMAL };
    StatusBar1->SetFieldsCount(1,__wxStatusBarWidths_1);
    StatusBar1->SetStatusStyles(1,__wxStatusBarStyles_1);
    SetStatusBar(StatusBar1);
    GridBagSizer1->Fit(this);
    GridBagSizer1->SetSizeHints(this);

    Connect(ID_RETEXT1,wxEVT_COMMAND_TEXT_UPDATED,(wxObjectEventFunction)&complex_setFrame::OnReText1Text);
    Connect(ID_IMTEXT1,wxEVT_COMMAND_TEXT_UPDATED,(wxObjectEventFunction)&complex_setFrame::OnImText1Text);
    Connect(ID_NEW1,wxEVT_COMMAND_BUTTON_CLICKED,(wxObjectEventFunction)&complex_setFrame::OnNew1Click);
    Connect(ID_SAVE1,wxEVT_COMMAND_BUTTON_CLICKED,(wxObjectEventFunction)&complex_setFrame::OnSave1Click);
    Connect(ID_DELETE1,wxEVT_COMMAND_BUTTON_CLICKED,(wxObjectEventFunction)&complex_setFrame::OnDelete1Click);
    Connect(ID_RETEXT2,wxEVT_COMMAND_TEXT_UPDATED,(wxObjectEventFunction)&complex_setFrame::OnReText2Text);
    Connect(ID_IMTEXT2,wxEVT_COMMAND_TEXT_UPDATED,(wxObjectEventFunction)&complex_setFrame::OnImText2Text);
    Connect(ID_NEW2,wxEVT_COMMAND_BUTTON_CLICKED,(wxObjectEventFunction)&complex_setFrame::OnNew2Click);
    Connect(ID_SAVE2,wxEVT_COMMAND_BUTTON_CLICKED,(wxObjectEventFunction)&complex_setFrame::OnSave2Click);
    Connect(ID_DELETE2,wxEVT_COMMAND_BUTTON_CLICKED,(wxObjectEventFunction)&complex_setFrame::OnDelete2Click);
    Connect(ID_LISTBOX1,wxEVT_COMMAND_LISTBOX_SELECTED,(wxObjectEventFunction)&complex_setFrame::OnSet1Select);
    Connect(ID_LISTBOX2,wxEVT_COMMAND_LISTBOX_SELECTED,(wxObjectEventFunction)&complex_setFrame::OnSet2Select);
    Connect(ID_MENUITEM1,wxEVT_COMMAND_MENU_SELECTED,(wxObjectEventFunction)&complex_setFrame::OnQuit);
    Connect(idMenuAbout,wxEVT_COMMAND_MENU_SELECTED,(wxObjectEventFunction)&complex_setFrame::OnAbout);
    //*)

    DrawPanel->Bind(wxEVT_PAINT, &complex_setFrame::OnPanelPaint, this);
    DrawPanel->SetBackgroundStyle(wxBG_STYLE_PAINT); // Enable double-buffering

    set1.LoadFromFile("set1.txt");
    set2.LoadFromFile("set2.txt");
    UpdateSet1ListBox();
    UpdateSet2ListBox();

    DrawPanel->Refresh();

    // Dragging
    isDragging = false;
    panOffset = wxPoint(0, 0);
    DrawPanel->Bind(wxEVT_LEFT_DOWN, &complex_setFrame::OnMouseDown, this);
    DrawPanel->Bind(wxEVT_LEFT_UP, &complex_setFrame::OnMouseUp, this);
    DrawPanel->Bind(wxEVT_MOTION, &complex_setFrame::OnMouseMove, this);
}

complex_setFrame::~complex_setFrame()
{
    //(*Destroy(complex_setFrame)
    //*)
}

void complex_setFrame::OnClose(wxCloseEvent& event) {
    set1.SaveToFile("set1.txt");
    set2.SaveToFile("set2.txt");
    event.Skip();
}

void complex_setFrame::UpdateSet1ListBox() {
    Set1->Clear();
    for (const auto& elem : set1.GetElements()) {
        Set1->Append(wxString::Format("(%.2f, %.2f)", elem.real(), elem.imag()));
    }
}

void complex_setFrame::UpdateSet2ListBox() {
    Set2->Clear();
    for (const auto& elem : set2.GetElements()) {
        Set2->Append(wxString::Format("(%.2f, %.2f)", elem.real(), elem.imag()));
    }
}

void complex_setFrame::OnPanelPaint(wxPaintEvent& event) {
    wxBufferedPaintDC dc(DrawPanel);
    dc.Clear();

    // Dragging
    dc.SetDeviceOrigin(-panOffset.x, -panOffset.y);

    // Calculate bounds
    double reMin1, reMax1, imMin1, imMax1;
    double reMin2, reMax2, imMin2, imMax2;
    set1.GetBounds(reMin1, reMax1, imMin1, imMax1);
    set2.GetBounds(reMin2, reMax2, imMin2, imMax2);

    double globalReMin = std::min(reMin1, reMin2);
    double globalReMax = std::max(reMax1, reMax2);
    double globalImMin = std::min(imMin1, imMin2);
    double globalImMax = std::max(imMax1, imMax2);

    // Handle empty sets
    if (set1.IsEmpty() && set2.IsEmpty()) {
        globalReMin = -1; globalReMax = 1;
        globalImMin = -1; globalImMax = 1;
    }

    // Add 10% padding
    double reRange = globalReMax - globalReMin;
    double imRange = globalImMax - globalImMin;
    globalReMin -= reRange * 0.1;
    globalReMax += reRange * 0.1;
    globalImMin -= imRange * 0.1;
    globalImMax += imRange * 0.1;

    if (globalReMin == globalReMax) { globalReMin -= 1; globalReMax += 1; }
    if (globalImMin == globalImMax) { globalImMin -= 1; globalImMax += 1; }

    // Get panel size
    int width, height;
    DrawPanel->GetSize(&width, &height);

    // Calculate scaling
    double scaleX = width / (globalReMax - globalReMin);
    double scaleY = height / (globalImMax - globalImMin);
    double scale = std::min(scaleX, scaleY);

    // Origin at center
    int originX = width / 2;
    int originY = height / 2;

    auto toPanelX = [=](double re) { return originX + (re - globalReMin) * scale; };
    auto toPanelY = [=](double im) { return originY - (im - globalImMin) * scale; };

    // Draw axes
    dc.SetPen(wxPen(*wxBLACK, 1));
    dc.DrawLine(0, originY, width, originY);
    dc.DrawLine(originX, 0, originX, height);

    // Draw grid
    dc.SetPen(wxPen(*wxLIGHT_GREY, 1, wxPENSTYLE_DOT));
    for (int x = floor(globalReMin); x <= ceil(globalReMax); ++x) {
        int xPos = toPanelX(x);
        dc.DrawLine(xPos, 0, xPos, height);
    }
    for (int y = floor(globalImMin); y <= ceil(globalImMax); ++y) {
        int yPos = toPanelY(y);
        dc.DrawLine(0, yPos, width, yPos);
    }

    // Draw elements
    auto drawSet = [&](const ComplexSet& set, const wxColour& color) {
        dc.SetBrush(wxBrush(color));
        for (const auto& elem : set.GetElements()) {
            int x = toPanelX(elem.real());
            int y = toPanelY(elem.imag());
            dc.DrawCircle(x, y, 5);
        }
    };

    drawSet(set1, *wxBLUE);
    drawSet(set2, *wxRED);

    // Draw intersection in green
    dc.SetBrush(*wxGREEN_BRUSH);
    for (const auto& elem : GetIntersection()) {
        int x = toPanelX(elem.real());
        int y = toPanelY(elem.imag());
        dc.DrawCircle(x, y, 5);
    }

    // Draw bounding rectangles
    auto drawBounds = [&](const ComplexSet& set, const wxColour& color) {
        double reMin, reMax, imMin, imMax;
        set.GetBounds(reMin, reMax, imMin, imMax);
        if (set.IsEmpty()) return;

        int x = toPanelX(reMin);
        int y = toPanelY(imMax);
        int w = toPanelX(reMax) - x;
        int h = toPanelY(imMin) - y;

        dc.SetPen(wxPen(color, 2));
        dc.SetBrush(*wxTRANSPARENT_BRUSH);
        dc.DrawRectangle(x, y, w, h);
    };

    drawBounds(set1, *wxBLUE);
    drawBounds(set2, *wxRED);
}

std::vector<std::complex<double>> complex_setFrame::GetIntersection() const {
    std::vector<std::complex<double>> intersection;
    for (const auto& elem : set1.GetElements()) {
        if (set2.Contains(elem)) {
            intersection.push_back(elem);
        }
    }
    return intersection;
}

void complex_setFrame::OnQuit(wxCommandEvent& event)
{
    Close();
}

void complex_setFrame::OnAbout(wxCommandEvent& event)
{
    wxString msg = wxbuildinfo(long_f);
    wxMessageBox(msg, _("Welcome to Two Complex Sets"));
}

void complex_setFrame::OnSet1Select(wxCommandEvent& event) {
    int selection = event.GetInt();
    if (selection == wxNOT_FOUND) return;

    auto elements = set1.GetElements();
    if (selection < elements.size()) {
        ReText1->SetValue(wxString::Format("%g", elements[selection].real()));
        ImText1->SetValue(wxString::Format("%g", elements[selection].imag()));
    }
    Redraw();
}

void complex_setFrame::OnSet2Select(wxCommandEvent& event)
{
    int selection = event.GetInt();
    if (selection == wxNOT_FOUND) return;

    auto elements = set2.GetElements();
    if (selection < elements.size()) {
        ReText2->SetValue(wxString::Format("%g", elements[selection].real()));
        ImText2->SetValue(wxString::Format("%g", elements[selection].imag()));
    }
    Redraw();
}

void complex_setFrame::OnNew1Click(wxCommandEvent& event) {
    double re, im;
    if (!ReText1->GetValue().ToDouble(&re) || !ImText1->GetValue().ToDouble(&im)) {
        wxMessageBox("Invalid input!", "Error", wxOK | wxICON_ERROR);
        return;
    }
    std::complex<double> elem(re, im);
    if (set1.Add(elem)) {
        Redraw();
        RedrawListbox();
    } else {
        wxMessageBox("Element already exists!", "Warning", wxOK | wxICON_WARNING);
    }
}

void complex_setFrame::OnNew2Click(wxCommandEvent& event)
{
    double re, im;
    if (!ReText2->GetValue().ToDouble(&re) || !ImText2->GetValue().ToDouble(&im)) {
        wxMessageBox("Invalid input!", "Error", wxOK | wxICON_ERROR);
        return;
    }
    std::complex<double> elem(re, im);
    if (set2.Add(elem)) {
        Redraw();
        RedrawListbox();
    } else {
        wxMessageBox("Element already exists!", "Warning", wxOK | wxICON_WARNING);
    }
}

void complex_setFrame::OnDelete1Click(wxCommandEvent& event) {
    int selection = Set1->GetSelection();
    if (selection == wxNOT_FOUND) return;

    auto elements = set1.GetElements();
    if (selection < elements.size()) {
        set1.Remove(elements[selection]);
        Redraw();
        RedrawListbox();
    }
}

void complex_setFrame::OnDelete2Click(wxCommandEvent& event)
{
    int selection = Set2->GetSelection();
    if (selection == wxNOT_FOUND) return;

    auto elements = set2.GetElements();
    if (selection < elements.size()) {
        set2.Remove(elements[selection]);
        Redraw();
        RedrawListbox();
    }
}

void complex_setFrame::OnSave1Click(wxCommandEvent& event)
{
    set1.SaveToFile("set1.txt");
    Redraw();
    RedrawListbox();
}

void complex_setFrame::OnSave2Click(wxCommandEvent& event)
{
    set2.SaveToFile("set2.txt");
    Redraw();
    RedrawListbox();
}

void complex_setFrame::OnImText1Text(wxCommandEvent& event)
{
}

void complex_setFrame::OnReText1Text(wxCommandEvent& event)
{
}

void complex_setFrame::OnImText2Text(wxCommandEvent& event)
{
}

void complex_setFrame::OnReText2Text(wxCommandEvent& event)
{
}

void complex_setFrame::OnMouseDown(wxMouseEvent& event) {
    isDragging = true;
    dragStartPos = event.GetPosition();
    DrawPanel->CaptureMouse(); // Ensure mouse capture
}

void complex_setFrame::OnMouseUp(wxMouseEvent& event) {
    if (isDragging) {
        isDragging = false;
        DrawPanel->ReleaseMouse();
    }
}

void complex_setFrame::OnMouseMove(wxMouseEvent& event) {
    if (isDragging) {
        wxPoint currentPos = event.GetPosition();
        wxPoint delta = currentPos - dragStartPos;
        panOffset += delta;
        dragStartPos = currentPos;
        DrawPanel->Refresh(); // Redraw with new offset
    }
}

void complex_setFrame::Redraw()
{
    panOffset = wxPoint(0, 0);
    DrawPanel->Refresh();
}

void complex_setFrame::RedrawListbox()
{
    UpdateSet1ListBox();
    UpdateSet2ListBox();
}
